!
! Copyright (C) 2016 Andrea Dal Corso 
! This file is distributed under the terms of the
! GNU General Public License. See the file `License'
! in the root directory of the present distribution,
! or http://www.gnu.org/copyleft/gpl.txt .
!
PROGRAM bravais_lattices
!
!  This is a simple code that reads three direct lattice vectors and
!  finds ibrav and celldm of that lattice. It can be used to transform
!  any triplet of primitive vectors in the input needed by thermo_pw or
!  QE.
!  Alternatively it checks the routine find_ibrav of the 
!  thermo_pw module by generating all the Bravais lattice vectors 
!  and checking that the module identify correctly the bravais lattice, then
!  it applies an arbitrary rotation and again checks that 
!  the routine find_ibrav identify correctly the lattice.
!
USE kinds,            ONLY : DP
USE mp_global,        ONLY : mp_startup, mp_global_end
USE environment,      ONLY : environment_start, environment_end
USE rotate,           ONLY : find_rotation
USE lattices,         ONLY : find_ibrav_code, same_lattice, &
                             lattice_point_group, compute_omega
USE matrix_inversion, ONLY : invmat
USE io_global,        ONLY : stdout

IMPLICIT NONE

CHARACTER(LEN=9) :: code='Bravais'
CHARACTER(LEN=11) :: gname
REAL(DP), PARAMETER :: eps1=1.D-8, eps2=1.D-5
REAL(DP) :: celldm(6), celldm2(6), at(3,3), at1(3,3), at2(3,3), at3(3,3),   &
            omega, celldm_(6), ur(3,3), sr(3,3), sr1(3,3), sr2(3,3), term1, &
            term2, term3, ax(3), global_s(3,3), srs(3,3,48), tau(3,2),      &
            tau_save(3,2), ur2(3,3), global_s2(3,3), invur2(3,3), invs2(3,3), &
            maxdiff
LOGICAL :: equivalent, lsame
INTEGER :: ibrav, ibrav_output, ibrav_output2, ipol, ivec, jvec, jpol,      &
           group_code, nsym, work_choice, code_group_ext, stdin

CALL mp_startup (start_images=.TRUE.)
CALL environment_start(code)

stdin=5
WRITE(stdout,'(/,5x,"Choose what to do")')
WRITE(stdout,'(5x,"1) Identify the Bravais lattice")')
WRITE(stdout,'(5x,"2) Compare two Bravais lattices")')

READ(stdin,*) work_choice

IF (work_choice==1) THEN
   WRITE(stdout,'(5x,"Input at? ")')
   DO ivec=1,3
      READ(stdin,*) (at(ipol,ivec),ipol=1,3)
   END DO

   code_group_ext=0
   CALL find_ibrav_code(at(1,1), at(1,2), at(1,3), ibrav_output, celldm_, & 
                                       code_group_ext,  ur, global_s,.TRUE.)
   WRITE(stdout,'(/,5x,"ibrav= ", i5)') ibrav_output
   WRITE(stdout,'(5x,"celldm(1)= ", f13.8)') celldm_(1)
   IF (ABS(celldm_(2))>eps1) &
      WRITE(stdout,'(5x,"celldm(2)= ", f13.8)') celldm_(2)
   IF (ABS(celldm_(3))>eps1) &
      WRITE(stdout,'(5x,"celldm(3)= ", f13.8)') celldm_(3)
   IF (ABS(celldm_(4))>eps1) &
      WRITE(stdout,'(5x,"celldm(4)= ", f13.8)') celldm_(4)
   IF (ABS(celldm_(5))>eps1) &
      WRITE(stdout,'(5x,"celldm(5)= ", f13.8)') celldm_(5)
   IF (ABS(celldm_(6))>eps1) &
      WRITE(stdout,'(5x,"celldm(6)= ", f13.8)') celldm_(6)

   WRITE(stdout,'(/,5x,"Rotation of the lattice generated by celldm")')
   WRITE(stdout,'(5x,"with respect to this lattice:",/)')

   DO ipol=1,3
      WRITE(stdout,'(5x,3f15.9)') (global_s(jpol,ipol),jpol=1,3)
   ENDDO
   WRITE(stdout,'(/,5x,"Apply this matrix to all atomic (cartesian) &
                                                             &coordinates.")')
   WRITE(stdout,'(5x,"See the translate.x tool to apply a roto-translation.")')

   WRITE(stdout,'(5x,"The ur matrix is")')
   DO ipol=1,3
      WRITE(stdout,'(5x,3f15.7)') (ur(ipol,jpol),jpol=1,3)
   ENDDO

ELSEIF (work_choice==2) THEN
   WRITE(stdout,'(5x,"celldm(1) (set to 1.0 to use the at1)")')
   READ(stdin,*) celldm(1)
   WRITE(stdout,'(5x,"Input at? ")')
   DO ivec=1,3
      READ(stdin,*) (at(ipol,ivec),ipol=1,3)
   END DO
   at=at * celldm(1)

   WRITE(stdout,'(5x,"celldm(1) (set to 1.0 to use the at2)")')
   READ(stdin,*) celldm2(1)
   WRITE(stdout,'(5x,"Input at2? ")')
   DO ivec=1,3
      READ(stdin,*) (at2(ipol,ivec),ipol=1,3)
   END DO
   at2 = at2 * celldm2(1)

   code_group_ext=0
   CALL find_ibrav_code(at(1,1), at(1,2), at(1,3), ibrav_output, celldm_, &
                                  code_group_ext, ur, global_s,.FALSE.)

   code_group_ext=0
   CALL find_ibrav_code(at2(1,1), at2(1,2), at2(1,3), ibrav_output2, celldm2, &
                                  code_group_ext, ur2, global_s2,.FALSE.)

   lsame = (ibrav_output==ibrav_output2)
   maxdiff=MAX(ABS(celldm_(1)-celldm2(1)), ABS(celldm_(2)-celldm2(2)), &
               ABS(celldm_(2)-celldm2(2)), ABS(celldm_(4)-celldm2(4)), &
               ABS(celldm_(3)-celldm2(3)), ABS(celldm_(6)-celldm2(6)) )
   lsame = lsame.AND.(maxdiff<eps2) 

   IF (lsame) THEN

      WRITE(stdout,'(/,5x,"The two lattices are the same within",e13.3)') &
                                                                 maxdiff
      WRITE(stdout,'(/,5x,"ibrav= ", i5)') ibrav_output
      WRITE(stdout,'("celldm =",6f12.8)') celldm_
      WRITE(stdout,'("celldm2=",6f12.8)') celldm2
      WRITE(stdout,'("diff   =",6f12.8)') celldm_-celldm2
      WRITE(stdout,'(/,5x,"at=S at1")')
      WRITE(stdout,'(5x,"at1=ur at2")')
!
!   Compute the rotation matrix
!
      CALL invmat(3, global_s2, invs2)
      global_s2=MATMUL(global_s, invs2)
!
!   Compute the ur matrix
!
      CALL invmat(3, ur2, invur2)
      ur2=MATMUL(invur2,ur)

      WRITE(stdout,'(5x,"The matrix S is:",/)')
      DO ipol=1,3
         WRITE(stdout,'(5x,3f15.9)') (global_s2(jpol,ipol),jpol=1,3)
      ENDDO

      CALL invmat(3, ur2, invur2)
      WRITE(stdout,'(/,5x,"The ur matrix is",23x,"The inverse of ur is",/)')
      DO ipol=1,3
         WRITE(stdout,'(5x,3f10.3,5x,3f10.3)') (ur2(ipol,jpol),jpol=1,3),     &
                                                (invur2(ipol,jpol),jpol=1,3)
      ENDDO

   ELSE
      WRITE(stdout,'(5x, "The two lattices are different")') 
      WRITE(stdout,'(5x, "The at have ")') 
      WRITE(stdout,'(/,5x,"ibrav= ", i5)') ibrav_output
      WRITE(stdout,'(5x,"celldm= ", 6f13.8)') celldm_
      WRITE(stdout,'(5x, "The at2 have ")') 
      WRITE(stdout,'(/,5x,"ibrav= ", i5)') ibrav_output2
      WRITE(stdout,'(5x,"celldm= ", 6f13.8)') celldm2
   ENDIF
ENDIF

CALL environment_end(code)
CALL mp_global_end ()

END PROGRAM bravais_lattices

